<?php

namespace Modules\Contacts\Models;

use Exception;
use App\Models\Config;
use App\Models\Company;
use App\Scopes\CompanyScope;
use Modules\Wpbox\Models\Reply;
use Modules\Wpbox\Models\Message;
use Modules\Wpbox\Models\Campaign;
use Modules\Wpbox\Traits\Whatsapp;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Http;
use Modules\Wpbox\Events\AgentReplies;
use Illuminate\Database\Eloquent\Model;
use Modules\Wpbox\Events\Chatlistchange;
use Modules\Wpbox\Events\ContactReplies;
use Illuminate\Database\Eloquent\SoftDeletes;
use Modules\Wpbox\Http\Controllers\HelpersChatController;
use Carbon\Carbon;

class Contact extends Model
{
    use SoftDeletes;
    use Whatsapp;

    protected $table = 'contacts';
    public $guarded = [];
    protected $fillable = [
        'name',
        'phone',
        'avatar',
        'country_id',
        'company_id',
        'last_reply_at',
        'last_client_reply_at',
        'last_support_reply_at',
        'last_message',
        'is_last_message_by_contact',
        'has_chat',
        'resolved_chat',
        'user_id',
        'enabled_ai_bot',
        'enable_auto_responses',
        'proceso_realizando',
        'confirmacionProceso',
        'pidiendo',
        'last_user_interaction',
        'ultimo_grupo_registrado',
    ];
    public function groups()
    {
        return $this->belongsToMany(
            Group::class,
            'groups_contacts',
            'contact_id',
            'group_id'
        );
    }

    public function getCompany()
    {
        return Company::first();
    }

    public function fields()
    {
        return $this->belongsToMany(
            Field::class,
            'custom_contacts_fields_contacts',
            'contact_id',
            'custom_contacts_field_id'
        )->withPivot('value');
    }

    public function messages()
    {
        return $this->hasMany(
            Message::class
        )->orderBy('created_at', 'DESC');
    }

    public function country()
    {
        return $this->belongsTo(
            Country::class
        );
    }


    private function trimString($str, $maxLength)
    {
        if (mb_strlen($str) <= $maxLength) {
            return $str;
        } else {
            $trimmed = mb_substr($str, 0, $maxLength);
            $lastSpaceIndex = mb_strrpos($trimmed, ' ');

            if ($lastSpaceIndex !== false) {
                return mb_substr($trimmed, 0, $lastSpaceIndex) . '...';
            } else {
                return $trimmed . '...';
            }
        }
    }




    public function sendDemoMessage($content, $is_message_by_contact = true, $is_campaign_messages = false, $messageType = "TEXT", $fb_message_id = null)
    {
        //Check that all is set ok

        //Create the messages
        $messageToBeSend = Message::create([
            "contact_id" => $this->id,
            "company_id" => $this->company_id,
            "value" => $messageType == "TEXT" ? $content : "",
            "header_image" => $messageType == "IMAGE" ? $content : "",
            "header_document" => $messageType == "DOCUMENT" ? $content : "",
            "header_video" => $messageType == "VIDEO" ? $content : "",
            "header_location" => $messageType == "LOCATION" ? $content : "",
            "is_message_by_contact" => $is_message_by_contact,
            "is_campign_messages" => $is_campaign_messages,
            "status" => 1,
            "buttons" => "[]",
            "components" => "",
            "fb_message_id" => $fb_message_id
        ]);
        $messageToBeSend->save();
    }



    /**
     * $reply - Reply - The reply to be send
     */
    public function sendReply(Reply $reply)
    {
        //Create the message
        $buttons = [];

        for ($i = 1; $i < 4; $i++) {
            if ($reply->{"button" . $i} != "") {
                $buttons[sizeof($buttons)] = [
                    "type" => "reply",
                    "reply" => [
                        "id" => $reply->{"button" . $i . "_id"},
                        "title" => $reply->{"button" . $i}
                    ]
                ];
            }
        }


        //If buttons is empty array
        $is_cta = false;
        if (sizeof($buttons) == 0) {
            //Check if we have set and not empty button_name and button_url
            if ($reply->button_name && $reply->button_name != "" && $reply->button_url && $reply->button_url != "") {
                $is_cta = true;
                $buttons[0] = [
                    "name" => "cta_url",
                    "parameters" => [
                        "display_text" => $reply->button_name,
                        "url" => $reply->button_url
                    ]
                ];
            }

        }



        $messageToBeSend = Message::create([
            "contact_id" => $this->id,
            "company_id" => $this->company_id,
            "value" => $reply->text,
            "header_text" => $reply->header,
            "footer_text" => $reply->footer,
            "buttons" => json_encode($buttons),
            "is_message_by_contact" => false,
            "is_campign_messages" => false,
            "status" => 1,
            "fb_message_id" => null
        ]);
        $messageToBeSend->save();
        if ($is_cta) {
            $messageToBeSend->is_cta = true;
        }


        $this->last_support_reply_at = now();
        $this->is_last_message_by_contact = false;
        $this->sendMessageToWhatsApp($messageToBeSend, $this);
        //Find the user of the company
        $companyUser = Company::first()->user;
        event(new AgentReplies($companyUser, $messageToBeSend, $this));

        $this->last_message = $this->trimString($reply->text, 40);
        $this->update();

        return $messageToBeSend;

    }


    public function botReply($content, $messageToBeSend)
    {
        //Reply bot
        $contact = Contact::where('id', $messageToBeSend['contact_id'])->get()->first();
        if ($contact->eneable_auto_responses) {
            // HelpersChatController::enviarConsultores($messageToBeSend['contact_id']);

            if (isset($this->last_user_interaction)) {
                $lastInteraction = $this->last_user_interaction instanceof Carbon
                    ? $this->last_user_interaction
                    : Carbon::parse($this->last_user_interaction);
                if ($lastInteraction->diffInMinutes(now()) > 5) {
                    HelpersChatController::enviarOpciones($messageToBeSend['contact_id']);
                }
            } else {
                HelpersChatController::enviarOpciones($messageToBeSend['contact_id']);
            }
        } else {
            return;
        }
        // $textReplies = Reply::where('type', '!=', 1)->where('company_id', $this->company_id)->get();
        // $replySend = false;
        // foreach ($textReplies as $key => $qr) {
        //     if (!$replySend) {
        // $replySend = $qr->shouldWeUseIt($content, $this);
        //     }
        // }

        // //If no text reply found, look for campaign reply
        // if (!$replySend) {
        //     $campaignReplies = Campaign::where('is_bot', 1)->where('is_bot_active', 1)->where('company_id', $this->company_id)->get();
        //     foreach ($campaignReplies as $key => $cr) {
        //         if (!$replySend) {
        //             $replySend = $cr->shouldWeUseIt($content, $this);
        //         }
        //     }
        // }

        // if ($replySend) {
        //     $messageToBeSend->bot_has_replied = true;
        //     $messageToBeSend->update();
        // }

    }



    /**
     * $content- String - The content to be stored, text or link
     * $is_message_by_contact - Boolean - is this a message send by contact - RECEIVE
     * $is_campaign_messages - Boolean - is this a message generated from campaign
     * $messageType [TEXT | IMAGE | VIDEO | DOCUMENT ]
     * $fb_message_id String - The Facebook message ID
     */
    public function sendMessage($content, $is_message_by_contact = true, $is_campaign_messages = false, $messageType = "TEXT", $fb_message_id = null)
    {
        //Check that all is set ok

        //If message is from contact, and fb_message_id is set, check if the message is already in the system
        if ($is_message_by_contact && $fb_message_id) {
            $message = Message::where('fb_message_id', $fb_message_id)->first();
            if ($message) {
                return $message;
            }
        }

        //Create the messages
        $messageToBeSend = Message::create([
            "contact_id" => $this->id,
            "company_id" => 1,
            "value" => $messageType == "TEXT" ? $content : "",
            "header_image" => $messageType == "IMAGE" ? $content : "",
            "header_document" => $messageType == "DOCUMENT" ? $content : "",
            "header_video" => $messageType == "VIDEO" ? $content : "",
            "header_audio" => $messageType == "AUDIO" ? $content : "",
            "header_location" => $messageType == "LOCATION" ? $content : "",
            "is_message_by_contact" => $is_message_by_contact,
            "is_campign_messages" => $is_campaign_messages,
            "status" => 1,
            "buttons" => "[]",
            "components" => "",
            "fb_message_id" => $fb_message_id
        ]);
        $messageToBeSend->save();



        //Update the contact last message, time etc


        if (!$is_campaign_messages) {
            $this->has_chat = true;
            $this->last_reply_at = now();
            if ($is_message_by_contact) {
                $this->last_client_reply_at = now();
                $this->is_last_message_by_contact = true;

                //Reply bots

                $this->botReply($content, $messageToBeSend);

                //Notify
                event(new ContactReplies(auth()->user(), $messageToBeSend, $this));
                event(new Chatlistchange($this->id, 1));


            } else {
                try {
                    $consultor = auth()->user();
                    $this->last_user_interaction = now();
                    $this->save();
                    if (!auth()->user()->hasRole('owner')) {
                        if (!$consultor->hasRole('staff') && $consultor->id !== $this->user_id) {
                            throw new Exception('[contact]');
                        }
                    }

                    $this->last_support_reply_at = now();
                    $this->is_last_message_by_contact = false;
                    $this->sendMessageToWhatsApp($messageToBeSend, $this);
                    event(new AgentReplies(auth()->user(), $messageToBeSend, $this));
                } catch (Exception $e) {
                    \Log::debug('error', ['error' => $e->getMessage()]);
                }
            }
        }
        $this->last_message = $this->trimString($content, 40);
        $this->update();





        return $messageToBeSend;
    }

    protected static function booted()
    {
        static::addGlobalScope(new CompanyScope);

        static::creating(function ($model) {
            $company_id = session('company_id', null);
            if ($company_id) {
                $model->company_id = 1;
            }
        });

        static::created(function ($model) {
            //Determine the country
            $country_id = $model->getCountryByPhoneNumber($model->phone);
            if ($country_id) {
                $model->country_id = $country_id;
                $model->update();
            }

        });
    }

    public function responseBot()
    {
        if (!isset($this->last_user_interaction)) {
            return true;
        }

        // Convertir `last_user_interaction` a un objeto Carbon si es un string
        $lastInteraction = $this->last_user_interaction instanceof Carbon
            ? $this->last_user_interaction
            : Carbon::parse($this->last_user_interaction);

        // Verificar si han pasado más de 5 minutos
        if ($lastInteraction->diffInMinutes(now()) > 5) {
            return true;
        }

        return false;
    }

    private function getCountryByPhoneNumber($phoneNumber)
    {

        if (strpos($phoneNumber, '+') !== 0) {
            $phoneNumber = '+' . $phoneNumber;
        }

        $prefixes = Country::pluck('id', 'phone_code');

        // Use regular expression to extract the prefix
        if (preg_match('/^\+(\d{1})/', $phoneNumber, $matches)) {
            $prefix = $matches[1];

            // Check if the prefix exists in the array
            if (isset($prefixes[$prefix])) {
                return $prefixes[$prefix];
            } else if (preg_match('/^\+(\d{2})/', $phoneNumber, $matches)) {
                $prefix = $matches[1];

                // Check if the prefix exists in the array
                if (isset($prefixes[$prefix])) {
                    return $prefixes[$prefix];
                } else if (preg_match('/^\+(\d{3})/', $phoneNumber, $matches)) {
                    $prefix = $matches[1];

                    // Check if the prefix exists in the array
                    if (isset($prefixes[$prefix])) {
                        return $prefixes[$prefix];
                    }
                }
            }
        }

        return null;
    }
}
