<?php

namespace Modules\Contacts\Http\Controllers;

use App\Http\Helpers\ApiResponse;
use App\Jobs\ImportContactsJob;
use Illuminate\Support\Facades\Validator;
use Modules\Contacts\Models\Contact;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use App\Http\Controllers\Controller;
use App\Models\Plans;
use App\Models\User;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Storage;
use Maatwebsite\Excel\Facades\Excel;
use Modules\Contacts\Exports\ContactsExport;
use Modules\Contacts\Imports\ContactsImport;
use Modules\Contacts\Models\Country;
use Modules\Contacts\Models\Field;
use Modules\Contacts\Models\Group;
use Illuminate\Support\Facades\Session;

class Main extends Controller
{
    /**
     * Provide class.
     */
    private $provider = Contact::class;

    /**
     * Web RoutePath for the name of the routes.
     */
    private $webroute_path = 'contacts.';

    /**
     * View path.
     */
    private $view_path = 'contacts::contacts.';

    /**
     * Parameter name.
     */
    private $parameter_name = 'contact';

    /**
     * Title of this crud.
     */
    private $title = 'contact';

    /**
     * Title of this crud in plural.
     */
    private $titlePlural = 'contacts';

    public function insertContactWithGroup(Request $request)
    {
        $token = $request->header('token');

        if (!isset($token)) {
            return response()->json([
                'status' => 'BAD_REQUEST',
            ], 400);
        }

        if ($token != '9c0407d3337b790a1829efe4ef59a33182190092e229904e31e1d4f17bf64ce1')
            return response()->json([
                'status' => 'BAD_REQUEST',
            ], 400);

        $params = json_decode($request->getContent(), true);

        if (empty($params))
            return response()->json([
                'status' => 'BAD_REQUEST',
            ], 400);

        $validator = Validator::make($params, [
            'telefono' => 'required|string|max:10',
            'nombre' => 'required|string|max:100',
            'grupo' => 'required|string|max:100',
        ]);

        if ($validator->fails())
            return response()->json([
                'status' => 'BAD_REQUEST',
            ], 400);

        $params = json_decode(json_encode($params)); // USE PARAMS AS AN OBJECT

        try {
            $colonia = Group::where('name', $params->grupo)->first();

            if (!isset($colonia)) {
                return response()->json([
                    'status' => 'BAD_REQUEST',
                ], 400);
            }

            $contact = Contact::updateOrCreate(
                [
                    'phone' => '521' . $params->telefono
                ],
                [
                    'phone' => '521' . $params->telefono,
                    'name' => $params->nombre,
                    'company_id' => 1
                ]
            );

            $contact->groups()->sync([$colonia->id]);

            return response()->json([
                'status' => 'SUCCESS',
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'error' => $e,
                'status' => 'ERROR',
            ], 500);
        }
    }

    private function hasAccessToAIBots()
    {
        $allowedPluginsPerPlan = auth()->user()->company ? auth()->user()->company->getPlanAttribute()['allowedPluginsPerPlan'] : [];
        if ($allowedPluginsPerPlan == null || in_array("flowiseai", $allowedPluginsPerPlan)) {
            return true;
        } else {
            return false;
        }
    }

    private function getFields($class = 'col-md-4', $getCustom = true)
    {
        $fields = [];

        //Avatar
        $fields[0] = ['class' => $class, 'ftype' => 'image', 'name' => 'Avatar', 'id' => 'avatar', 'style' => 'width: 200px; height:200'];

        //Add name field
        $fields[1] = ['class' => $class, 'ftype' => 'input', 'name' => 'Name', 'id' => 'name', 'placeholder' => 'Enter name', 'required' => true];

        //Add phone field
        $fields[2] = ['class' => $class, 'ftype' => 'input', 'type' => "phone", 'name' => 'Phone', 'id' => 'phone', 'placeholder' => 'Enter phone', 'required' => true];

        //Groups
        $fields[3] = ['class' => $class, 'multiple' => true, 'classselect' => "select2init", 'ftype' => 'select', 'name' => 'Groups', 'id' => 'groups[]', 'placeholder' => 'Select group', 'data' => Group::get()->pluck('name', 'id'), 'required' => true];

        //Country
        // $fields[4] = ['class' => $class, 'ftype' => 'select', 'name' => 'Country', 'id' => 'country_id', 'placeholder' => 'Select country', 'data' => Country::get()->pluck('name', 'id'), 'required' => true];

        //AI Bot enabled
        $customFieldStart = 4;

        if ($this->hasAccessToAIBots()) {
            $customFieldStart = 5;
            $fields[4] = ['class' => $class, 'ftype' => 'bool', 'name' => 'Enable AI bot Replies', 'id' => 'enabled_ai_bot', 'placeholder' => 'AI Bot replies enabled', 'required' => false];

        }


        if ($getCustom) {
            $customFields = Field::get()->toArray();
            $i = $customFieldStart;
            foreach ($customFields as $filedkey => $customField) {
                $i++;
                $fields[$i] = ['class' => $class, 'ftype' => 'input', 'type' => $customField['type'], 'name' => __($customField['name']), 'id' => "custom[" . $customField['id'] . "]", 'placeholder' => __($customField['name']), 'required' => false];

            }
        }


        //Return fields
        return $fields;
    }


    private function getFilterFields()
    {
        $fields = $this->getFields('col-md-3', false);
        unset($fields[0]);
        $fields[1]['required'] = false;
        $fields[2]['required'] = false;

        $fields[3]['required'] = false;
        $fields[3]['multiple'] = false;
        $fields[3]['id'] = 'group';
        unset($fields[3]['multiple']);

        // $fields[4]['required'] = false;
        // $fields[4]['multiple'] = false;
        // unset($fields[4]['multiple']);

        unset($fields[5]);

        //unset($fields[2]);
        return $fields;
    }

    /**
     * Auth checker functin for the crud.
     */
    private function authChecker()
    {
        $this->ownerAndStaffOnly();
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $this->authChecker();

        $items = $this->provider::orderBy('id', 'desc');
        if (isset($_GET['name']) && strlen($_GET['name']) > 1) {
            $items = $items->where('name', 'like', '%' . $_GET['name'] . '%');
        }
        if (isset($_GET['phone']) && strlen($_GET['phone']) > 1) {
            $items = $items->where('phone', 'like', '%' . $_GET['phone'] . '%');
        }

        if (isset($_GET['group']) && strlen($_GET['group'] . "") > 0) {
            $items = $items->whereHas('groups', function ($query) {
                $query->where('groups.id', $_GET['group']);
            });
        }
        if (isset($_GET['country_id']) && strlen($_GET['country_id']) > 0) {
            $items = $items->where('country_id', $_GET['country_id']);
        }

        if (isset($_GET['report'])) {
            //dd($items->with(['fields','groups'])->get());
            return $this->exportCSV($items->with(['fields', 'groups'])->get());

        }
        $totalItems = $items->count();
        $items = $items->paginate(config('settings.paginate'));


        return view($this->view_path . 'index', [
            'setup' => [
                'usefilter' => true,
                'title' => __('crud.item_managment', ['item' => __($this->titlePlural)]),
                'subtitle' => $totalItems == 1 ? __('1 Contact') : $totalItems . " " . __('Contacts'),
                'action_link' => route($this->webroute_path . 'create'),
                'action_name' => __('crud.add_new_item', ['item' => __($this->title)]),
                'action_link2' => route($this->webroute_path . 'groups.index'),
                'action_name2' => __('Groups'),
                'action_link3' => route($this->webroute_path . 'fields.index'),
                'action_name3' => __('Fields'),
                'action_link4' => route($this->webroute_path . 'index', ['report' => true]),
                'action_name4' => __('Export'),
                'items' => $items,
                'item_names' => $this->titlePlural,
                'webroute_path' => $this->webroute_path,
                'fields' => $this->getFields(),
                'filterFields' => $this->getFilterFields(),
                'custom_table' => true,
                'parameter_name' => $this->parameter_name,
                'parameters' => count($_GET) != 0,
                'groups' => Group::get(),
            ]
        ]);
    }

    public function exportCSV($contactsToDownload)
    {
        $items = [];
        $cf = Field::get();
        foreach ($contactsToDownload as $key => $contact) {
            $item = [
                'id' => $contact->id,
                'name' => $contact->name,
                'phone' => $contact->phone,
                'avatar' => $contact->avatar,
            ];

            foreach ($cf as $keycf => $scf) {
                $item[$scf->name] = "";
                foreach ($contact->fields as $key => $value) {
                    if ($scf->name == $value['name']) {
                        $item[$value['name']] = $value['pivot']['value'];
                    }

                }
            }


            array_push($items, $item);
        }
        return Excel::download(new ContactsExport($items), 'contacts_' . time() . '.csv', \Maatwebsite\Excel\Excel::CSV);
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $this->authChecker();


        return view($this->view_path . 'edit', [
            'setup' => [
                'title' => __('crud.new_item', ['item' => __($this->title)]),
                'action_link' => route($this->webroute_path . 'index'),
                'action_name' => __('crud.back'),
                'iscontent' => true,
                'action' => route($this->webroute_path . 'store'),
            ],
            'fields' => $this->getFields()
        ]);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $this->authChecker();

        //Create new contact
        $phone = preg_replace('/[^0-9]/', '', $request->phone);
        $phone = str_replace(' ', '', $phone);
        $contact = $this->provider::create([
            'name' => $request->name,
            'phone' => $phone,
        ]);
        $contact->save();

        if ($request->has('avatar')) {
            if (config('settings.use_s3_as_storage', false)) {
                //S3
                $contact->avatar = Storage::disk('s3')->url($request->avatar->storePublicly("uploads/" . $contact->company_id . "/contacts", 's3'));
            } else {
                $contact->avatar = Storage::disk('public_media_upload')->url($request->avatar->store(null, 'public_media_upload'));
            }

            $contact->update();
        }

        // Attaching groups to the contact
        $contact->groups()->attach($request->groups);

        if (isset($request->custom)) {
            $this->syncCustomFieldsToContact($request->custom, $contact);
        }

        return redirect()->route($this->webroute_path . 'index')->withStatus(__('crud.item_has_been_added', ['item' => __($this->title)]));
    }



    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Contact  $contacts
     * @return \Illuminate\Http\Response
     */
    public function edit(Contact $contact)
    {
        $this->authChecker();

        $fields = $this->getFields();
        $fields[0]['value'] = $contact->avatar;
        $fields[1]['value'] = $contact->name;
        $fields[2]['value'] = $contact->phone;

        $fields[3]['multipleselected'] = $contact->groups->pluck('id')->toArray();
        // $fields[4]['value'] = $contact->country_id;

        if ($this->hasAccessToAIBots()) {
            $fields[4]['value'] = $contact->enabled_ai_bot . "" == "1";
        }


        $customFieldsValues = $contact->fields->toArray();
        foreach ($customFieldsValues as $key => $fieldWithPivot) {
            foreach ($fields as $key => &$formField) {
                if ($formField['id'] == "custom[" . $fieldWithPivot['id'] . "]") {
                    $formField['value'] = $fieldWithPivot['pivot']['value'];
                }
            }
        }


        $parameter = [];
        $parameter[$this->parameter_name] = $contact->id;

        return view($this->view_path . 'edit', [
            'setup' => [
                'title' => __('crud.edit_item_name', ['item' => __($this->title), 'name' => $contact->name]),
                'action_link' => route($this->webroute_path . 'index'),
                'action_name' => __('crud.back'),
                'iscontent' => true,
                'isupdate' => true,
                'action' => route($this->webroute_path . 'update', $parameter),
            ],
            'fields' => $fields,
        ]);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Contact  $contacts
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $this->authChecker();
        $phone = preg_replace('/[^0-9]/', '', $request->phone);
        $phone = str_replace(' ', '', $phone);
        $item = $this->provider::findOrFail($id);
        $item->name = $request->name;
        $item->phone = $phone;
        if ($this->hasAccessToAIBots()) {
            $item->enabled_ai_bot = $request->enabled_ai_bot == "true";
        }


        if ($request->has('avatar')) {
            if (config('settings.use_s3_as_storage', false)) {
                //S3
                $item->avatar = Storage::disk('s3')->url($request->avatar->storePublicly("uploads/" . $item->company_id . "/contacts", 's3'));
            } else {
                $item->avatar = Storage::disk('public_media_upload')->url($request->avatar->store(null, 'public_media_upload'));
            }


        }

        $item->update();

        if (isset($request->custom)) {
            $this->syncCustomFieldsToContact($request->custom, $item);
        }


        // Attaching groups to the contact
        $item->groups()->sync($request->groups);
        $item->update();

        return redirect()->route($this->webroute_path . 'index')->withStatus(__('crud.item_has_been_updated', ['item' => __($this->title)]));
    }


    public function syncCustomFieldsToContact($fields, $contact)
    {
        $contact->fields()->sync([]);
        foreach ($fields as $key => $value) {
            if ($value) {
                $contact->fields()->attach($key, ['value' => $value]);
            }

        }
        $contact->update();
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Contact  $contacts
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        $this->authChecker();
        $item = $this->provider::findOrFail($id);
        $item->delete();
        return redirect()->route($this->webroute_path . 'index')->withStatus(__('crud.item_has_been_removed', ['item' => __($this->title)]));
    }

    public function bulkremove($ids)
    {
        $this->authChecker();
        $ids = explode(",", $ids);
        $this->provider::destroy($ids);

        // Return a JSON response
        return response()->json([
            'status' => 'success',
            'message' => __('crud.items_have_been_removed', ['item' => __($this->titlePlural)])
        ], 200);
    }

    public function assigntogroup($ids)
    {
        $this->authChecker();
        $ids = explode(",", $ids);
        $group = Group::find($_GET['group_id']);

        if (!$group) {
            // Group not found, return an error response
            return response()->json([
                'status' => 'error',
                'message' => __('No group selected')
            ], 404);
        }

        $group->contacts()->syncWithoutDetaching($ids);

        // Return a JSON response
        return response()->json([
            'status' => 'success',
            'message' => __('crud.items_has_been_updated', ['item' => __($this->titlePlural)])
        ], 200);
    }

    public function removefromgroup($ids)
    {
        $this->authChecker();
        $ids = explode(",", $ids);
        $group = Group::find($_GET['group_id']);

        if (!$group) {
            // Group not found, return an error response
            return response()->json([
                'status' => 'error',
                'message' => __('No group selected')
            ], 404);
        }

        $group->contacts()->detach($ids);

        // Return a JSON response
        return response()->json([
            'status' => 'success',
            'message' => __('crud.items_has_been_updated', ['item' => __($this->titlePlural)])
        ], 200);
    }

    public function importindex()
    {
        $groups = Group::pluck('name', 'id');
        return view("contacts::" . $this->webroute_path . 'import', ['groups' => $groups]);
    }

    public function import(Request $request)
    {
        $request->validate([
            'xlsx' => 'required|file|mimes:xlsx',
            'group' => 'nullable|integer|exists:groups,id'
        ]);

        $file = $request->file('xlsx');
        $group = $request->input('group', null);
        $userId = auth()->user()->id;

        // Almacenar el archivo
        $filePath = $file->store('imports');

        // Despachar el Job
        ImportContactsJob::dispatch($filePath, $group, $userId);

        return response()->json(['status' => 'success', 'message' => __('Import process started.')], 200);
    }

    public function getImportProgress()
    {
        $userId = auth()->user()->id;
        $progress = Session::get('import_progress_' . $userId, 0);
        $errors = Session::get('import_errors_' . $userId, []);

        return response()->json(['progress' => $progress, 'errors' => $errors]);
    }
    public function importContactEcommerce(Request $request)
    {
        try {
            if ($request->clave != '60S];^PK~f0V') {
                return response()->json([
                    'status' => 'Error',
                    'message' => 'No puedes accesar',
                    'dataset' => 'No puedes accesar'
                ], 400);
            }
            // Validar los datos del request sin la regla unique
            $validator = Validator::make($request->all(), [
                'celular' => 'string|max:10|min:10|required',
                'nombre' => 'string|max:191|required',
            ], [
                'celular.required' => 'El celular es obligatorio',
                'celular.max' => 'El celular debe contener 10 caracteres',
                'celular.min' => 'El celular debe contener 10 caracteres',
                'celular.string' => 'El celular debe ser tipo string',
                'nombre.required' => 'El nombre es obligatorio',
                'nombre.max' => 'El nombre solo puede tener 191 caracteres',
                'nombre.string' => 'El nombre debe ser un string',
            ]);

            // Si la validación falla, retornar una respuesta con los errores
            if ($validator->fails()) {
                return response()->json([
                    'status' => 'Error',
                    'message' => $validator->errors()->first(),
                    'dataset' => $validator->errors()
                ], 400);
            }
            $group_id = 4;
            // Verificar si ya existe un contacto con el número de celular
            $contact = Contact::where('phone', '521' + $request->celular)->first();

            if ($contact) {
                $group = \DB::selectOne('Select * from groups_contacts where contact_id = ?', [$contact->id]);
                if ($group) {
                    \DB::update('update groups_contacts set group_id = ? where contact_id = ?', [$group_id, $contact->id]);
                } else {
                    \DB::insert('insert into groups_contacts (contact_id, group_id) values ( ? , ? )', [$contact->id, $group_id]);
                    return response()->json([
                        'status' => 'Success',
                        'message' => 'OK',
                        'data' => $contact
                    ], 200);
                }
            } else {
                // Si no existe, crear un nuevo contacto
                $newContact = Contact::create([
                    "name" => $request->nombre,
                    "phone" => '521' + $request->celular,
                    "country_id" => 375,
                    "company_id" => 1,
                ]);

                \DB::insert('insert into groups_contacts (contact_id, group_id) values ( ? , ? )', [$newContact->id, $group_id]);

                // Retornar una respuesta indicando que el contacto fue creado
                return response()->json([
                    'status' => 'Success',
                    'message' => 'El contacto ha sido creado con éxito.',
                    'data' => $newContact
                ], 201);
            }
        } catch (\Exception $e) {
            \Log::debug(json_encode($e->getMessage()));

            return response()->json([
                'status' => 'Error',
                'message' => $e->getMessage(),
                'dataset' => $e->getMessage()
            ], 500);
        }
    }
    public function importContactsEcommerce(Request $request)
    {
        try {
            // Validar que el request contenga un arreglo de contactos
            $validator = Validator::make($request->all(), [
                'contacts' => 'required|array',
                'contacts.*.celular' => 'string|max:13|min:13|required',
                'contacts.*.nombre' => 'string|max:191|required',
            ], [
                'contacts.required' => 'Se requiere un arreglo de contactos.',
                'contacts.*.celular.required' => 'El celular es obligatorio',
                'contacts.*.celular.max' => 'El celular debe contener 13 caracteres',
                'contacts.*.celular.min' => 'El celular debe contener 13 caracteres',
                'contacts.*.celular.string' => 'El celular debe ser tipo string',
                'contacts.*.nombre.required' => 'El nombre es obligatorio',
                'contacts.*.nombre.max' => 'El nombre solo puede tener 191 caracteres',
                'contacts.*.nombre.string' => 'El nombre debe ser un string',
            ]);

            // Si la validación falla, retornar una respuesta con los errores
            if ($validator->fails()) {
                return response()->json([
                    'status' => 'Error',
                    'message' => 'Validación fallida',
                    'dataset' => $validator->errors()
                ], 400);
            }

            // Inicializar contadores
            $successCount = 0;
            $errorCount = 0;
            $group_id = 4;

            // Recorrer cada contacto en el arreglo
            foreach ($request->contacts as $contactData) {
                try {
                    // Verificar si ya existe un contacto con el número de celular
                    $contact = Contact::where('phone', $contactData['celular'])->first();

                    if ($contact) {
                        // Verificar si el contacto ya está asignado a un grupo
                        $group = \DB::selectOne('Select * from groups_contacts where contact_id = ?', [$contact->id]);
                        if ($group) {
                            \DB::update('update groups_contacts set group_id = ? where contact_id = ?', [$group_id, $contact->id]);
                        } else {
                            \DB::insert('insert into groups_contacts (contact_id, group_id) values ( ? , ? )', [$contact->id, $group_id]);
                        }

                        // Incrementar el contador de éxito
                        $successCount++;
                    } else {
                        // Si no existe, crear un nuevo contacto
                        $newContact = Contact::create([
                            "name" => $contactData['nombre'],
                            "phone" => $contactData['celular'],
                            "country_id" => 375,
                            "company_id" => 1,
                        ]);

                        \DB::insert('insert into groups_contacts (contact_id, group_id) values ( ? , ? )', [$newContact->id, $group_id]);

                        // Incrementar el contador de éxito
                        $successCount++;
                    }
                } catch (\Exception $e) {
                    // Si ocurre un error con este contacto, incrementar el contador de errores y continuar con el siguiente
                    \Log::error('Error al procesar el contacto: ' . json_encode($contactData));
                    \Log::error($e->getMessage());
                    $errorCount++;
                }
            }

            // Retornar los resultados del proceso
            return response()->json([
                'status' => 'Success',
                'message' => 'Procesamiento completado',
                'insertados_actualizados' => $successCount,
                'errores' => $errorCount
            ], 200);

        } catch (\Exception $e) {
            \Log::error($e->getMessage());

            return response()->json([
                'status' => 'Error',
                'message' => $e->getMessage(),
                'dataset' => $e->getMessage()
            ], 500);
        }
    }





}
