<?php

namespace App\Jobs;

use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Maatwebsite\Excel\Facades\Excel;
use Modules\Contacts\Models\Contact;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\Storage;

class ImportContactsJob implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    protected $filePath;
    protected $group;
    protected $userId;

    public function __construct($filePath, $group, $userId)
    {
        $this->filePath = $filePath;
        $this->group = $group;
        $this->userId = $userId;
    }

    public function handle()
    {
        $errors = [];

        try {
            $file = Storage::path($this->filePath);
            $contacts = Excel::toCollection(new \Modules\Contacts\Imports\ContactsImport, $file);
            $totalContacts = count($contacts[0]);
            $processedContacts = 0;

            foreach ($contacts[0] as $contactData) {
                try {
                    // Limpiar el número de teléfono
                    if (!$contactData['phone'] || $contactData['phone'] == '') {
                        throw new \Exception('No existe número de telefono.');
                    }
                    $phone = preg_replace('/[^0-9]/', '', $contactData['phone']);
                    $phone = str_replace(' ', '', $contactData['phone']);
                    if (strlen($phone) === 10) {
                        $phone = '521' . $phone;
                    } elseif (strlen($phone) === 13 && substr($phone, 0, 3) === '521') {
                        // No hacer nada
                    } elseif (strlen($phone) === 12 && substr($phone, 0, 2) === '52') {
                        $phone = '521' . substr($phone, 2);
                    } else {
                        throw new \Exception('Número de teléfono no válido.');
                    }

                    // Capitalizar el nombre
                    $name = ucwords(strtolower($contactData['name']));

                    // Verificar si el contacto ya existe
                    $existingContact = Contact::where('phone', $phone)->first();
                    if ($existingContact) {
                        continue;
                    }

                    // Crear el nuevo contacto
                    $newContact = Contact::create([
                        'name' => $name,
                        'phone' => $phone,
                    ]);

                    // Añadir el contacto al grupo si es necesario
                    if ($this->group) {
                        $newContact->groups()->attach($this->group);
                    }

                } catch (\Exception $e) {
                    $errors[] = [
                        'name' => $contactData['name'],
                        'phone' => $contactData['phone'],
                        'error' => $e->getMessage()
                    ];
                }

                $processedContacts++;
                $progress = ($processedContacts / $totalContacts) * 100;
                Session::put('import_progress_' . $this->userId, $progress);
            }

            Session::put('import_errors_' . $this->userId, $errors);
            // Asegurarse de que el progreso se establezca en 100%
            Session::put('import_progress_' . $this->userId, 100);

        } catch (\Exception $e) {
            \Log::error('Error importing contacts: ' . $e->getMessage());
        } finally {
            // Eliminar el archivo después de la importación
            Storage::delete($this->filePath);
        }
    }
}
