<?php
namespace App\Exports;

use Modules\ReportesBot\Models\Reporte;
use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithMapping;
use Maatwebsite\Excel\Concerns\WithDrawings;
use Maatwebsite\Excel\Concerns\ShouldAutoSize;
use PhpOffice\PhpSpreadsheet\Worksheet\Drawing;
use Illuminate\Support\Facades\Storage;

class ReportesExport implements FromCollection, WithHeadings, WithMapping, WithDrawings, ShouldAutoSize
{
    private $reportes;
    private $tempFiles = [];

    private $filters;


    public function __construct($filters = [])
    {
        $this->filters = $filters;

        // Aplicar filtros a la consulta
        $query = Reporte::query();


        if (!empty($filters['folio'])) {
            $query->where('folio', 'LIKE', "%{$filters['folio']}%");
        }

        if (!empty($filters['tipo_reporte'])) {
            $query->where('tipo_reporte', 'LIKE', "%{$filters['tipo_reporte']}%");
        }

        if (!empty($filters['nombre_contacto'])) {
            $query->where('nombre_contacto', 'LIKE', "%{$filters['nombre_contacto']}%");
        }

        if (!empty($filters['ubicacion'])) {
            $query->where('ubicacion', 'LIKE', "%{$filters['ubicacion']}%");
        }

        if (!empty($filters['fecha_inicio']) && !empty($filters['fecha_fin'])) {
            $query->whereBetween('created_at', [$filters['fecha_inicio'], $filters['fecha_fin']]);
        }

        if (!empty($filters['estatus'])) {
            $query->where('estatus', $filters['estatus']);
        }

        if (!empty($filters['estatusOpciones'])) {
            $query->where('estatusOpciones', $filters['estatusOpciones']);
        }

        // Obtener reportes filtrados
        $this->reportes = $query->get()->map(function ($reporte) {
            $reporte->id = 'R' . $reporte->id;

            // Convertir estatusOpciones a texto
            $estatus = [
                1 => 'Creado',
                2 => 'Capturado',
                3 => 'En ejecución',
                4 => 'Terminado',
                5 => 'Crítico',
            ];
            $reporte->estatusOpciones = $estatus[$reporte->estatusOpciones] ?? 'Desconocido';

            return $reporte;
        });
    }

    public function collection()
    {
        return $this->reportes;
    }

    public function headings(): array
    {
        return [
            'ID',
            'Tipo de Reporte',
            'Celular',
            'Nombre o Servicio',
            'Descripción',
            'Ubicación',
            'Referencia',
            'Estatus',
            'Foto del Reporte',
            'Foto del Contexto',
            'Fecha de Creación',
        ];
    }

    public function map($reporte): array
    {
        return [
            $reporte->id,
            $reporte->tipo_reporte,
            $reporte->celular,
            $reporte->nombre_o_servicio,
            $reporte->descripcion_reporte,
            $reporte->ubicacion,
            $reporte->referencia,
            $reporte->estatusOpciones,
            '',// $reporte->foto_reporte,
            '',// $reporte->foto_context_lugar,
            $reporte->created_at->format('d/m/Y'),
        ];
    }

    public function drawings()
    {
        $drawings = [];
        $startRow = 2; // Comienza después de los encabezados

        foreach ($this->reportes as $index => $reporte) {
            $currentRow = $startRow + $index;

            // Foto del reporte
            if (!empty($reporte->foto_reporte)) {
                $photoPath = $this->downloadImage($reporte->foto_reporte);
                if ($photoPath) {
                    $drawing = new Drawing();
                    $drawing->setName('Foto del Reporte');
                    $drawing->setDescription('Foto del Reporte');
                    $drawing->setPath($photoPath); // Ruta de la imagen descargada
                    $drawing->setHeight(50); // Altura de la imagen
                    $drawing->setCoordinates('I' . $currentRow); // Columna para la foto
                    $drawings[] = $drawing;
                }
            }

            // Foto del contexto
            if (!empty($reporte->foto_context_lugar)) {
                $photoContextPath = $this->downloadImage($reporte->foto_context_lugar);
                if ($photoContextPath) {
                    $drawing = new Drawing();
                    $drawing->setName('Foto del Contexto');
                    $drawing->setDescription('Foto del Contexto');
                    $drawing->setPath($photoContextPath); // Ruta de la imagen descargada
                    $drawing->setHeight(50); // Altura de la imagen
                    $drawing->setCoordinates('J' . $currentRow); // Columna para el contexto
                    $drawings[] = $drawing;
                }
            }
        }

        return $drawings;
    }

    private function downloadImage($url)
    {
        try {
            // Generar un archivo temporal para la imagen
            $contents = file_get_contents($url);
            $tempPath = tempnam(sys_get_temp_dir(), 'excel_img_') . '.jpg';
            file_put_contents($tempPath, $contents);

            // Guardar la referencia al archivo temporal
            $this->tempFiles[] = $tempPath;

            return $tempPath;
        } catch (\Exception $e) {
            return null;
        }
    }

    public function __destruct()
    {
        // Limpiar archivos temporales después de generar el Excel
        foreach ($this->tempFiles as $tempFile) {
            if (file_exists($tempFile)) {
                unlink($tempFile);
            }
        }
    }
}
